//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQml 2.12
import QtQuick 2.12

import wesual.Ui 1.0
import wesual.Ui.Private 1.0

Rectangle {
    id : uiCalendar

    property date date
    property date minimumDate
    property date maximumDate

    property bool bindable : false

    signal updateRequested(date newValue)

    function reset() {
        content.selectedDate = new Date(Number.NaN);
    }

    function dateInBounds(date, minDate, maxDate) {
        if (minDate.getFullYear() > 0) {
            if (date.getFullYear() < minDate.getFullYear()) {
                return false;
            } else if (date.getFullYear() === minDate.getFullYear()) {
                if (date.getMonth() < minDate.getMonth()) {
                    return false;
                } else  if (date.getMonth() === minDate.getMonth()) {
                    if (date.getDate() < minDate.getDate())
                        return false;
                }
            }
        }

        if (maxDate.getFullYear() > 0) {
            if (date.getFullYear() > maxDate.getFullYear()) {
                return false;
            } else if (date.getFullYear() === maxDate.getFullYear()) {
                if (date.getMonth() > maxDate.getMonth()) {
                    return false;
                } else  if (date.getMonth() === maxDate.getMonth()) {
                    if (date.getDate() > maxDate.getDate())
                        return false;
                }
            }
        }

        return true;
    }

    implicitWidth  : 240
    implicitHeight : content.implicitHeight + 2
    onDateChanged  : reset()

    border {
        width : 1
        color : UiColors.getColor(UiColors.ControlGrey20)
    }

    MouseArea {
        id : mouseArea

        anchors.fill : parent
        hoverEnabled : true
    }

    Component {
        id : dayDelegate

        Rectangle {
            id : dayField

            readonly property bool selected :
                date.getFullYear() === uiCalendar.date.getFullYear() &&
                date.getMonth() === uiCalendar.date.getMonth() &&
                date.getDate() === uiCalendar.date.getDate()

            readonly property date date : modelData
            readonly property bool today : {
                var now = new Date();
                return date.getFullYear() === now.getFullYear() &&
                        date.getMonth() === now.getMonth() &&
                        date.getDate() === now.getDate();
            }

            width   : 30
            height  : 22
            enabled : dateInBounds(date, uiCalendar.minimumDate,
                                   uiCalendar.maximumDate)

            border {
                color : UiColors.getColor(UiColors.DarkerSelectionGreen)
                width : dayField.today ? 1 : 0
            }

            states : [
                State {
                    name : "disabled-selected"
                    when : !dayField.enabled && dayField.selected

                    PropertyChanges {
                        target : dayField
                        color  : UiColors.getColor(UiColors.DisabledLightGrey13)
                    }
                    PropertyChanges {
                        target : label
                        color  : UiColors.getColor(UiColors.White)
                    }
                },
                State {
                    name : "disabled"
                    when : !dayField.enabled

                    PropertyChanges {
                        target : label
                        color  : UiColors.getColor(UiColors.MediumLightGrey27)
                    }
                },
                State {
                    name : "selected"
                    when : dayField.selected || trigger.containsPress

                    PropertyChanges {
                        target : dayField
                        color  : UiColors.getColor(UiColors.SelectionGreen)
                    }
                    PropertyChanges {
                        target : label
                        color  : UiColors.getColor(UiColors.White)
                    }
                },
                State {
                    name : "hovered"
                    when : trigger.containsMouse

                    PropertyChanges {
                        target : dayField
                        color  : UiColors.getColor(UiColors.HoverGreen)
                    }
                }
            ]

            MouseArea {
                id : trigger

                anchors.fill : parent
                hoverEnabled : true
                onPressed    : content.setDate(modelData)
            }

            Text {
                id : label

                text : modelData.getDate()
                font : UiFonts.getFont(UiFonts.Light, 12)
                anchors {
                    horizontalCenter : parent.horizontalCenter
                    bottom           : parent.bottom
                    bottomMargin     : 3
                }
            }
        }
    }

    Column {
        id : content

        readonly property bool dateValid         : date.getFullYear() > 0
        readonly property bool selectedDateValid :
            selectedDate.getFullYear() > 0

        readonly property var locale : Qt.locale()

        property date selectedDate

        property date displayDate : {
            if (selectedDate.getYear() > 0) {
                return selectedDate;
            } else if (content.dateValid) {
                return uiCalendar.date;
            }

            return new Date();
        }

        function incrementMonth() {
            var date = displayDate;
            date.setMonth(date.getMonth() + 1);
            selectedDate = date;
        }

        function decrementMonth() {
            var date = displayDate;
            date.setMonth(date.getMonth() - 1);
            selectedDate = date;
        }

        function setDate(date) {
            date.setHours(0);
            date.setMinutes(0);
            date.setSeconds(0);
            date.setMilliseconds(0);

            if (uiCalendar.bindable) {
                uiCalendar.updateRequested(date);
            } else {
                uiCalendar.date = date;
            }
        }

        anchors.fill    : parent
        anchors.margins : 1

        UiCalendarInfo {
            id : calendarInfo

            date : content.displayDate
        }

        Item {
            height        : 32
            anchors.left  : parent.left
            anchors.right : parent.right

            UiToolButton {
                iconName : "left"
                iconSize : UiIcon.Icon_16x16
                anchors {
                    left   : parent.left
                    top    : parent.top
                    bottom : parent.bottom
                }
                onClicked : content.decrementMonth()
            }

            UiText {
                id : month

                anchors.centerIn : parent
                text : content.locale.monthName(
                           content.displayDate.getMonth(),
                           Locale.LongFormat).toUpperCase() + " " +
                       content.displayDate.getFullYear()
            }

            UiToolButton {
                iconName : "right"
                iconSize : UiIcon.Icon_16x16
                anchors {
                    right  : parent.right
                    top    : parent.top
                    bottom : parent.bottom
                }
                onClicked : content.incrementMonth()
            }
        }
        Rectangle {
            height        : 1
            anchors.left  : parent.left
            anchors.right : parent.right
            color         : UiColors.getColor(UiColors.ControlGrey20)
        }
        Item {
            height        : 165
            anchors.left  : parent.left
            anchors.right : parent.right

            Row {
                id : dayNames

                Repeater {
                    model : 7
                    delegate : UiSectionHeader {
                        width  : 30
                        height : 20
                        text   : content.locale.dayName(
                                     modelData + 1,
                                     Locale.ShortFormat).substr(0, 2)
                        font   : UiFonts.getFont(UiFonts.RegularCapitalized, 12)
                        verticalAlignment   : Qt.AlignVCenter
                        horizontalAlignment : Qt.AlignHCenter
                    }
                }
                anchors {
                    bottom : days.top
                    left   : days.left
                }
            }
            Grid {
                id : days

                columns : 7

                Repeater {
                    model : calendarInfo.days
                    delegate : dayDelegate
                }
                anchors {
                    bottom       : parent.bottom
                    right        : parent.right
                    bottomMargin : 6
                    rightMargin  : 3
                }
            }
            Column {
                id : weekNo

                property var weekList

                Repeater {
                    model : calendarInfo.calendarWeeks
                    delegate : Item {
                        width  : 24
                        height : 22

                        Text {
                            text  : modelData
                            font  : UiFonts.getFont(UiFonts.Regular, 10)
                            color : UiColors.getColor(UiColors.MediumGrey33)
                            anchors {
                                horizontalCenter : parent.horizontalCenter
                                bottom           : parent.bottom
                                bottomMargin     : 4
                            }
                        }
                    }
                }
                anchors {
                    right : days.left
                    top   : days.top
                }
            }
        }
        Rectangle {
            height        : 1
            anchors.left  : parent.left
            anchors.right : parent.right
            color         : UiColors.getColor(UiColors.ControlGrey20)
        }
        Item {
            height        : 38
            anchors.left  : parent.left
            anchors.right : parent.right

            UiCommandLink {
                text      : qsTrId("8298533db336ff8c")
                onClicked : content.setDate(new Date())
                anchors {
                    centerIn             : parent
                    verticalCenterOffset : 2
                }
                enabled   : dateInBounds(
                                new Date(), uiCalendar.minimumDate,
                                uiCalendar.maximumDate)
            }
        }
    }
}
